<?php
/**
 * ComponentSchema Model
 * Manages component configuration schemas
 */
namespace Models;

class ComponentSchema
{
    private static $pdo;

    public static function setPdo($pdo)
    {
        self::$pdo = $pdo;
    }

    /**
     * Initialize the component_schemas table
     */
    public static function createTable()
    {
        self::$pdo->exec("
            CREATE TABLE IF NOT EXISTS component_schemas (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                component_name TEXT UNIQUE NOT NULL,
                display_name TEXT NOT NULL,
                category TEXT DEFAULT 'general',
                icon TEXT DEFAULT 'fas fa-puzzle-piece',
                description TEXT,
                schema TEXT NOT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )
        ");
    }

    /**
     * Get all schemas
     */
    public static function all()
    {
        $stmt = self::$pdo->query("SELECT * FROM component_schemas ORDER BY category, display_name");
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }

    /**
     * Get schema by component name
     */
    public static function get($componentName)
    {
        $stmt = self::$pdo->prepare("SELECT * FROM component_schemas WHERE component_name = ?");
        $stmt->execute([$componentName]);
        $result = $stmt->fetch(\PDO::FETCH_ASSOC);
        if ($result && isset($result['schema'])) {
            $result['schema'] = json_decode($result['schema'], true);
        }
        return $result;
    }

    /**
     * Get schemas by category
     */
    public static function getByCategory($category)
    {
        $stmt = self::$pdo->prepare("SELECT * FROM component_schemas WHERE category = ? ORDER BY display_name");
        $stmt->execute([$category]);
        return $stmt->fetchAll(\PDO::FETCH_ASSOC);
    }

    /**
     * Create or update schema
     */
    public static function upsert($data)
    {
        $schema = is_array($data['schema']) ? json_encode($data['schema']) : $data['schema'];

        $stmt = self::$pdo->prepare("
            INSERT INTO component_schemas (component_name, display_name, category, icon, description, schema)
            VALUES (?, ?, ?, ?, ?, ?)
            ON CONFLICT(component_name) DO UPDATE SET
                display_name = excluded.display_name,
                category = excluded.category,
                icon = excluded.icon,
                description = excluded.description,
                schema = excluded.schema,
                updated_at = CURRENT_TIMESTAMP
        ");

        return $stmt->execute([
            $data['component_name'],
            $data['display_name'],
            $data['category'] ?? 'general',
            $data['icon'] ?? 'fas fa-puzzle-piece',
            $data['description'] ?? '',
            $schema
        ]);
    }

    /**
     * Delete schema
     */
    public static function delete($componentName)
    {
        $stmt = self::$pdo->prepare("DELETE FROM component_schemas WHERE component_name = ?");
        return $stmt->execute([$componentName]);
    }

    /**
     * Get all categories
     */
    public static function getCategories()
    {
        $stmt = self::$pdo->query("SELECT DISTINCT category FROM component_schemas ORDER BY category");
        return $stmt->fetchAll(\PDO::FETCH_COLUMN);
    }

    /**
     * Generate default schemas for existing components
     */
    public static function seedDefaultSchemas()
    {
        $defaults = self::getDefaultSchemas();
        foreach ($defaults as $schema) {
            self::upsert($schema);
        }
    }

    /**
     * Default schemas for built-in components
     */
    public static function getDefaultSchemas()
    {
        return [
            // HERO COMPONENTS
            [
                'component_name' => 'hero',
                'display_name' => 'Hero Section',
                'category' => 'headers',
                'icon' => 'fas fa-heading',
                'description' => 'Full-width hero section with title, subtitle and CTA',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık', 'required' => true, 'placeholder' => 'Ana Başlık'],
                        ['name' => 'subtitle', 'type' => 'textarea', 'label' => 'Alt Başlık', 'rows' => 2],
                        ['name' => 'cta_text', 'type' => 'text', 'label' => 'Buton Metni'],
                        ['name' => 'cta_link', 'type' => 'url', 'label' => 'Buton Linki'],
                        [
                            'name' => 'background',
                            'type' => 'select',
                            'label' => 'Arkaplan Rengi',
                            'options' => [
                                ['value' => 'primary', 'label' => 'Primary (Mavi)'],
                                ['value' => 'info', 'label' => 'Info (Açık Mavi)'],
                                ['value' => 'success', 'label' => 'Success (Yeşil)'],
                                ['value' => 'warning', 'label' => 'Warning (Sarı)'],
                                ['value' => 'danger', 'label' => 'Danger (Kırmızı)'],
                                ['value' => 'dark', 'label' => 'Dark (Siyah)'],
                                ['value' => 'light', 'label' => 'Light (Açık)']
                            ]
                        ],
                        [
                            'name' => 'size',
                            'type' => 'select',
                            'label' => 'Boyut',
                            'options' => [
                                ['value' => 'medium', 'label' => 'Orta'],
                                ['value' => 'large', 'label' => 'Büyük'],
                                ['value' => 'fullheight', 'label' => 'Tam Ekran']
                            ]
                        ]
                    ]
                ]
            ],

            // CONTENT COMPONENTS
            [
                'component_name' => 'features',
                'display_name' => 'Features Grid',
                'category' => 'content',
                'icon' => 'fas fa-th-large',
                'description' => 'Feature boxes with icons',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                        ['name' => 'subtitle', 'type' => 'textarea', 'label' => 'Alt Başlık'],
                        [
                            'name' => 'columns',
                            'type' => 'select',
                            'label' => 'Kolon Sayısı',
                            'options' => [
                                ['value' => '3', 'label' => '3 Kolon'],
                                ['value' => '4', 'label' => '4 Kolon'],
                                ['value' => '2', 'label' => '2 Kolon']
                            ]
                        ],
                        [
                            'name' => 'items',
                            'type' => 'repeater',
                            'label' => 'Özellikler',
                            'fields' => [
                                ['name' => 'icon', 'type' => 'icon', 'label' => 'İkon'],
                                ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                                ['name' => 'description', 'type' => 'textarea', 'label' => 'Açıklama']
                            ]
                        ]
                    ]
                ]
            ],

            [
                'component_name' => 'content_section',
                'display_name' => 'Content Section',
                'category' => 'content',
                'icon' => 'fas fa-align-left',
                'description' => 'Two-column content with image',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık', 'required' => true],
                        ['name' => 'content', 'type' => 'richtext', 'label' => 'İçerik'],
                        ['name' => 'image', 'type' => 'media', 'label' => 'Görsel'],
                        [
                            'name' => 'image_position',
                            'type' => 'select',
                            'label' => 'Görsel Pozisyonu',
                            'options' => [
                                ['value' => 'right', 'label' => 'Sağda'],
                                ['value' => 'left', 'label' => 'Solda']
                            ]
                        ],
                        ['name' => 'cta_text', 'type' => 'text', 'label' => 'Buton Metni'],
                        ['name' => 'cta_link', 'type' => 'url', 'label' => 'Buton Linki']
                    ]
                ]
            ],

            // PRICING
            [
                'component_name' => 'pricing',
                'display_name' => 'Pricing Table',
                'category' => 'commerce',
                'icon' => 'fas fa-tags',
                'description' => 'Pricing plans comparison',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                        [
                            'name' => 'plans',
                            'type' => 'repeater',
                            'label' => 'Planlar',
                            'fields' => [
                                ['name' => 'name', 'type' => 'text', 'label' => 'Plan Adı'],
                                ['name' => 'price', 'type' => 'text', 'label' => 'Fiyat'],
                                ['name' => 'period', 'type' => 'text', 'label' => 'Periyot', 'placeholder' => '/ay'],
                                ['name' => 'features', 'type' => 'textarea', 'label' => 'Özellikler (satır başına bir)'],
                                ['name' => 'cta_text', 'type' => 'text', 'label' => 'Buton Metni'],
                                ['name' => 'cta_link', 'type' => 'url', 'label' => 'Buton Linki'],
                                ['name' => 'is_featured', 'type' => 'checkbox', 'label' => 'Öne Çıkan']
                            ]
                        ]
                    ]
                ]
            ],

            // TESTIMONIALS
            [
                'component_name' => 'testimonials',
                'display_name' => 'Testimonials',
                'category' => 'social-proof',
                'icon' => 'fas fa-quote-left',
                'description' => 'Customer testimonials',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                        [
                            'name' => 'testimonials',
                            'type' => 'repeater',
                            'label' => 'Yorumlar',
                            'fields' => [
                                ['name' => 'quote', 'type' => 'textarea', 'label' => 'Yorum'],
                                ['name' => 'author', 'type' => 'text', 'label' => 'İsim'],
                                ['name' => 'role', 'type' => 'text', 'label' => 'Pozisyon'],
                                ['name' => 'image', 'type' => 'media', 'label' => 'Fotoğraf']
                            ]
                        ]
                    ]
                ]
            ],

            // FAQ
            [
                'component_name' => 'faq',
                'display_name' => 'FAQ Accordion',
                'category' => 'content',
                'icon' => 'fas fa-question-circle',
                'description' => 'Frequently asked questions',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                        [
                            'name' => 'faqs',
                            'type' => 'repeater',
                            'label' => 'Sorular',
                            'fields' => [
                                ['name' => 'question', 'type' => 'text', 'label' => 'Soru'],
                                ['name' => 'answer', 'type' => 'textarea', 'label' => 'Cevap']
                            ]
                        ]
                    ]
                ]
            ],

            // TEAM
            [
                'component_name' => 'team',
                'display_name' => 'Team Members',
                'category' => 'content',
                'icon' => 'fas fa-users',
                'description' => 'Team member cards',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                        ['name' => 'description', 'type' => 'textarea', 'label' => 'Açıklama'],
                        [
                            'name' => 'members',
                            'type' => 'repeater',
                            'label' => 'Üyeler',
                            'fields' => [
                                ['name' => 'name', 'type' => 'text', 'label' => 'İsim'],
                                ['name' => 'role', 'type' => 'text', 'label' => 'Pozisyon'],
                                ['name' => 'image', 'type' => 'media', 'label' => 'Fotoğraf'],
                                ['name' => 'bio', 'type' => 'textarea', 'label' => 'Biyografi']
                            ]
                        ]
                    ]
                ]
            ],

            // CONTACT
            [
                'component_name' => 'contact_form',
                'display_name' => 'Contact Form',
                'category' => 'forms',
                'icon' => 'fas fa-envelope',
                'description' => 'Contact form with fields',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                        ['name' => 'subtitle', 'type' => 'textarea', 'label' => 'Alt Başlık'],
                        ['name' => 'email_to', 'type' => 'email', 'label' => 'Alıcı E-posta'],
                        ['name' => 'success_message', 'type' => 'text', 'label' => 'Başarı Mesajı']
                    ]
                ]
            ],

            // NEWSLETTER
            [
                'component_name' => 'newsletter',
                'display_name' => 'Newsletter Signup',
                'category' => 'forms',
                'icon' => 'fas fa-paper-plane',
                'description' => 'Email subscription form',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                        ['name' => 'subtitle', 'type' => 'textarea', 'label' => 'Alt Başlık'],
                        ['name' => 'button_text', 'type' => 'text', 'label' => 'Buton Metni'],
                        [
                            'name' => 'background',
                            'type' => 'select',
                            'label' => 'Arkaplan',
                            'options' => [
                                ['value' => 'light', 'label' => 'Açık'],
                                ['value' => 'primary', 'label' => 'Primary'],
                                ['value' => 'dark', 'label' => 'Koyu']
                            ]
                        ]
                    ]
                ]
            ],

            // CTA
            [
                'component_name' => 'cta_banner',
                'display_name' => 'CTA Banner',
                'category' => 'marketing',
                'icon' => 'fas fa-bullhorn',
                'description' => 'Call-to-action banner',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık', 'required' => true],
                        ['name' => 'subtitle', 'type' => 'textarea', 'label' => 'Alt Başlık'],
                        ['name' => 'cta_text', 'type' => 'text', 'label' => 'Buton Metni'],
                        ['name' => 'cta_link', 'type' => 'url', 'label' => 'Buton Linki'],
                        [
                            'name' => 'background',
                            'type' => 'select',
                            'label' => 'Arkaplan',
                            'options' => [
                                ['value' => 'primary', 'label' => 'Primary'],
                                ['value' => 'info', 'label' => 'Info'],
                                ['value' => 'success', 'label' => 'Success'],
                                ['value' => 'dark', 'label' => 'Dark']
                            ]
                        ]
                    ]
                ]
            ],

            // STATS
            [
                'component_name' => 'stats',
                'display_name' => 'Statistics',
                'category' => 'social-proof',
                'icon' => 'fas fa-chart-bar',
                'description' => 'Counter statistics',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                        [
                            'name' => 'stats',
                            'type' => 'repeater',
                            'label' => 'İstatistikler',
                            'fields' => [
                                ['name' => 'value', 'type' => 'text', 'label' => 'Değer', 'placeholder' => '500+'],
                                ['name' => 'label', 'type' => 'text', 'label' => 'Etiket'],
                                ['name' => 'icon', 'type' => 'icon', 'label' => 'İkon']
                            ]
                        ]
                    ]
                ]
            ],

            // GALLERY
            [
                'component_name' => 'gallery',
                'display_name' => 'Image Gallery',
                'category' => 'media',
                'icon' => 'fas fa-images',
                'description' => 'Image gallery grid',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                        [
                            'name' => 'columns',
                            'type' => 'select',
                            'label' => 'Kolon Sayısı',
                            'options' => [
                                ['value' => '3', 'label' => '3 Kolon'],
                                ['value' => '4', 'label' => '4 Kolon'],
                                ['value' => '2', 'label' => '2 Kolon']
                            ]
                        ],
                        [
                            'name' => 'images',
                            'type' => 'repeater',
                            'label' => 'Görseller',
                            'fields' => [
                                ['name' => 'src', 'type' => 'media', 'label' => 'Görsel'],
                                ['name' => 'caption', 'type' => 'text', 'label' => 'Açıklama']
                            ]
                        ]
                    ]
                ]
            ],

            // VIDEO
            [
                'component_name' => 'video_embed',
                'display_name' => 'Video Embed',
                'category' => 'media',
                'icon' => 'fas fa-video',
                'description' => 'YouTube or Vimeo video',
                'schema' => [
                    'fields' => [
                        ['name' => 'url', 'type' => 'url', 'label' => 'Video URL', 'required' => true, 'placeholder' => 'https://youtube.com/watch?v=...'],
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                        ['name' => 'description', 'type' => 'textarea', 'label' => 'Açıklama'],
                        [
                            'name' => 'aspect',
                            'type' => 'select',
                            'label' => 'En-Boy Oranı',
                            'options' => [
                                ['value' => '16by9', 'label' => '16:9'],
                                ['value' => '4by3', 'label' => '4:3'],
                                ['value' => '1by1', 'label' => '1:1']
                            ]
                        ]
                    ]
                ]
            ],

            // MAP
            [
                'component_name' => 'map',
                'display_name' => 'Map (LeafletJS)',
                'category' => 'media',
                'icon' => 'fas fa-map-marked-alt',
                'description' => 'Interactive map with markers',
                'schema' => [
                    'fields' => [
                        ['name' => 'lat', 'type' => 'number', 'label' => 'Enlem (Latitude)', 'placeholder' => '41.0082'],
                        ['name' => 'lng', 'type' => 'number', 'label' => 'Boylam (Longitude)', 'placeholder' => '28.9784'],
                        ['name' => 'zoom', 'type' => 'number', 'label' => 'Zoom Seviyesi', 'placeholder' => '13'],
                        ['name' => 'height', 'type' => 'text', 'label' => 'Yükseklik', 'placeholder' => '400px'],
                        [
                            'name' => 'style',
                            'type' => 'select',
                            'label' => 'Harita Stili',
                            'options' => [
                                ['value' => 'streets', 'label' => 'Streets'],
                                ['value' => 'satellite', 'label' => 'Satellite'],
                                ['value' => 'terrain', 'label' => 'Terrain']
                            ]
                        ],
                        [
                            'name' => 'markers',
                            'type' => 'repeater',
                            'label' => 'İşaretçiler',
                            'fields' => [
                                ['name' => 'lat', 'type' => 'number', 'label' => 'Enlem'],
                                ['name' => 'lng', 'type' => 'number', 'label' => 'Boylam'],
                                ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                                ['name' => 'popup', 'type' => 'textarea', 'label' => 'Popup İçeriği']
                            ]
                        ]
                    ]
                ]
            ],

            // COUNTDOWN
            [
                'component_name' => 'countdown',
                'display_name' => 'Countdown Timer',
                'category' => 'marketing',
                'icon' => 'fas fa-clock',
                'description' => 'Countdown to date',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                        ['name' => 'target_date', 'type' => 'text', 'label' => 'Hedef Tarih', 'placeholder' => '2024-12-31 23:59:59'],
                        ['name' => 'end_message', 'type' => 'text', 'label' => 'Bitiş Mesajı'],
                        [
                            'name' => 'background',
                            'type' => 'select',
                            'label' => 'Arkaplan',
                            'options' => [
                                ['value' => 'dark', 'label' => 'Koyu'],
                                ['value' => 'primary', 'label' => 'Primary'],
                                ['value' => 'light', 'label' => 'Açık']
                            ]
                        ]
                    ]
                ]
            ],

            // SERVICES
            [
                'component_name' => 'services',
                'display_name' => 'Services Grid',
                'category' => 'content',
                'icon' => 'fas fa-concierge-bell',
                'description' => 'Service cards with icons',
                'schema' => [
                    'fields' => [
                        ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                        ['name' => 'subtitle', 'type' => 'textarea', 'label' => 'Alt Başlık'],
                        [
                            'name' => 'services',
                            'type' => 'repeater',
                            'label' => 'Hizmetler',
                            'fields' => [
                                ['name' => 'icon', 'type' => 'icon', 'label' => 'İkon'],
                                ['name' => 'title', 'type' => 'text', 'label' => 'Başlık'],
                                ['name' => 'description', 'type' => 'textarea', 'label' => 'Açıklama'],
                                ['name' => 'link', 'type' => 'url', 'label' => 'Link']
                            ]
                        ]
                    ]
                ]
            ]
        ];
    }
}
